from contextlib import contextmanager
import matplotlib.pyplot as plt
import torch.nn as nn
import torch

from core.components.base_model import BaseModel
from core.modules.losses import L2Loss
from core.modules.subnetworks import Encoder, Decoder, Predictor
from core.modules.recurrent_modules import RecurrentPredictor
from core.utils.general_utils import AttrDict, ParamDict, batch_apply
from core.utils.pytorch_utils import pad_seq
from core.modules.layers import LayerBuilderParams
from core.utils.vis_utils import make_gif_strip, make_image_seq_strip, make_image_strip

class MultiValueStateModel(BaseModel):
    def __init__(self, params, logger):
        super().__init__(logger)
        self._hp = self._default_hparams()
        self._hp.overwrite(params)
        self._hp.builder = LayerBuilderParams(self._hp.use_convs, self._hp.normalization)
        self._task_names = [task_name().name if not isinstance(task_name, str) else task_name
                              for task_name in self._hp.task_names]

        self.build_network()

    @contextmanager
    def val_mode(self):
        pass
        yield
        pass

    def _default_hparams(self):
        default_dict = ParamDict({
            'use_skips': False,
            'skips_stride': 2,
            'add_weighted_pixel_copy': False, # if True, adds pixel copying stream for decoder
            'pixel_shift_decoder': False,
            'use_convs': False,
            'detach_reconstruction': True,
            'detach_discounted_return_heads': False,
            'n_cond_frames': 1,
            'normalization': 'none'
        })

        # Network size
        default_dict.update({
            'img_sz': 32,
            'input_nc': 3,
            'ngf': 8,
            'nz_enc': 32,
            'nz_mid': 32,
            'n_processing_layers': 3,
            'n_pixel_sources': 1,
        })

        # Loss weights
        default_dict.update({
            'img_mse_weight': 1.,
            'reward_weights': 1.,
        })

        # add new params to parent params
        parent_params = super()._default_hparams()
        parent_params.overwrite(default_dict)
        return parent_params

    def build_network(self):
        self.encoder = Encoder(self._hp)
        self.decoder = Decoder(self._hp)
        self.discounted_return_heads = nn.ModuleDict({name: Predictor(self._hp, input_size=self._hp.nz_enc*self._hp.n_frames,
                                                                      output_size=1, spatial=False) for name in self._task_names})

    def forward(self, inputs):
        """
        forward pass at training time
        """
        output = AttrDict()

        # encode inputs
        enc = self.encoder(inputs.states)
        output.update({'pred': enc})

        rec_input = output.pred.detach() if self._hp.detach_reconstruction else output.pred
        output.output_imgs = self.decoder(rec_input, skips=skips).images.unsqueeze(1)

        # reward decoding
        discounted_return_input = output.pred.detach() if self._hp.detach_discounted_return_heads else output.pred
        output.discounted_returns = AttrDict({name: self.discounted_return_heads[name](discounted_return_input)
                                   for name in self._task_names})

        return output

    def loss(self, model_output, inputs):
        losses = AttrDict()

        # image reconstruction loss
        losses.seq_img_mse = L2Loss(self._hp.img_mse_weight)(model_output.output_imgs,
                                                             inputs.images[:, :, -3:])

        # reward regression loss
        losses.update(AttrDict({name: L2Loss(self._hp.reward_weights)(model_output.discounted_returns[name][inputs.task_id==i],
                                                                      inputs.discounted_returns[inputs.task_id==i])
                                for i, name in enumerate(self._task_names)}))

        losses.total = self._compute_total_loss(losses)
        return losses

    def log_outputs(self, model_output, inputs, losses, step, log_images, phase):
        super()._log_losses(losses, step, log_images, phase)
        if log_images:
            # log predicted images
            img_strip = make_image_strip([inputs.images.squeeze(1)[:, -3:], model_output.output_imgs.squeeze(1)])
            self._logger.log_images(img_strip[None], 'generation', step, phase)

            # log reward prediction vs time step
            for i, name in enumerate(self._task_names):
                loss_per_step = L2Loss()(model_output.discounted_returns[name][inputs.task_id==i],
                                                  inputs.discounted_returns[inputs.task_id==i])
                self._logger.log_scalar(loss_per_step, name+"_loss_breakdown", step, phase)

    def forward_encoder(self, inputs):
        enc, skips = self.encoder(inputs)
        return enc

    @property
    def resolution(self):
        return self._hp.img_sz


